<?php
session_start();
$page_title = "Menyu Kateqoriyaları";
require_once '../includes/header.php'; 

// Yalnız Adminlər bu səhifəyə daxil ola bilər (Əgər bu yoxdursa, əlavə edin)
if (!isset($_SESSION['role']) || $_SESSION['role'] !== 'admin') {
    header('Location: index.php');
    exit;
}

$message = $_GET['message'] ?? '';
$error = $_GET['error'] ?? '';

// --- 1. Silmə Əməliyyatı (DELETE) (Dəyişməyib) ---
if (isset($_GET['action']) && $_GET['action'] == 'delete' && isset($_GET['id'])) {
    $category_id = (int)$_GET['id'];
    
    // Qeyd: parent_id üzərində ON DELETE SET NULL olduğu üçün alt kateqoriyalar pozulmayacaq.
    $stmt = $conn->prepare("DELETE FROM categories WHERE id = ?");
    $stmt->bind_param("i", $category_id);
    
    if ($stmt->execute()) {
        $message = "Kateqoriya uğurla silindi.";
    } else {
        $error = "Silmək mümkün olmadı. Zəhmət olmasa, əvvəlcə bu kateqoriyaya bağlı **menyu məhsullarını** silin.";
    }
    $stmt->close();
    
    header('Location: categories.php?message=' . urlencode($message) . '&error=' . urlencode($error));
    exit;
}

// --- 2. Yaratma/Dəyişdirmə Əməliyyatı (CREATE / UPDATE) (YENİLƏNİB) ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name = trim($_POST['name']);
    $parent_id = !empty($_POST['parent_id']) ? (int)$_POST['parent_id'] : null; // Yeni
    $sort_order = (int)$_POST['sort_order']; // Yeni
    $is_active = isset($_POST['is_active']) ? 1 : 0; // Yeni
    $edit_id = (isset($_POST['edit_id']) && $_POST['edit_id'] != '') ? (int)$_POST['edit_id'] : null;

    if ($edit_id) {
        // UPDATE əməliyyatı
        $stmt = $conn->prepare("UPDATE categories SET name = ?, parent_id = ?, sort_order = ?, is_active = ? WHERE id = ?");
        // parent_id NULL ola biləcəyi üçün bind_param əməliyyatında 's' string kimi göndərilir, lakin SQL bunu NULL kimi qəbul edir
        // Alternativ: parent_id null-dırsa 'i' (integer) yerinə 's' (string) kimi null göndərilməlidir.
        $stmt->bind_param("siiii", $name, $parent_id, $sort_order, $is_active, $edit_id);
        $action_message = "Kateqoriya uğurla dəyişdirildi.";
    } else {
        // CREATE əməliyyatı
        $stmt = $conn->prepare("INSERT INTO categories (name, parent_id, sort_order, is_active) VALUES (?, ?, ?, ?)");
        $stmt->bind_param("siii", $name, $parent_id, $sort_order, $is_active);
        $action_message = "Yeni kateqoriya uğurla əlavə edildi.";
    }
    
    // Yoxlama: Kateqoriya özünü Ana kateqoriya kimi seçə bilməsin
    if ($edit_id && $parent_id && $edit_id == $parent_id) {
        $error = "Xəta: Bir kateqoriya özü-özünün Ana kateqoriyası ola bilməz!";
    }
    
    if (empty($error) && $stmt->execute()) {
        $message = $action_message;
    } elseif (empty($error)) {
        $error = "Əməliyyat uğursuz oldu: Kateqoriya adı təkrarlana bilər və ya verilənlər bazası səhvi.";
    }
    if (isset($stmt)) $stmt->close();
    
    header('Location: categories.php?message=' . urlencode($message) . '&error=' . urlencode($error));
    exit;
}

// --- 3. Dəyişdirmə üçün Datayı Yükləmə (READ Single) (YENİLƏNİB) ---
$category_to_edit = null;
if (isset($_GET['action']) && $_GET['action'] == 'edit' && isset($_GET['id'])) {
    $edit_id = (int)$_GET['id'];
    $stmt = $conn->prepare("SELECT id, name, parent_id, sort_order, is_active FROM categories WHERE id = ?");
    $stmt->bind_param("i", $edit_id);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($result->num_rows == 1) {
        $category_to_edit = $result->fetch_assoc();
    }
    $stmt->close();
}

// --- 4. Bütün Kateqoriyaların Siyahısı (READ All) (YENİLƏNİB) ---
// Siyahını çəkirik, Ana Kateqoriyanın adını da əlavə edirik
$categories = [];
$sql = "
    SELECT 
        c1.id, 
        c1.name, 
        c1.parent_id, 
        c1.sort_order, 
        c1.is_active, 
        c1.created_at,
        c2.name AS parent_name
    FROM categories c1
    LEFT JOIN categories c2 ON c1.parent_id = c2.id
    ORDER BY c1.sort_order ASC, c1.id ASC
";
$result = $conn->query($sql);
if ($result && $result->num_rows > 0) {
    while($row = $result->fetch_assoc()) {
        $categories[] = $row;
    }
}

// Parent ID seçimi üçün bütün kateqoriyalar (Özü daxil deyil)
$parent_options = [];
$parent_result = $conn->query("SELECT id, name FROM categories ORDER BY name ASC");
if ($parent_result) {
    while($row = $parent_result->fetch_assoc()) {
        if (!($category_to_edit && $category_to_edit['id'] == $row['id'])) { // Redaktə edilən kateqoriyanı seçimdən çıxar
            $parent_options[] = $row;
        }
    }
}

?>

<?php if ($message): ?>
    <div class="alert alert-success alert-dismissible fade show" role="alert"><i class="fas fa-check-circle me-2"></i> <?= htmlspecialchars($message) ?><button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button></div>
<?php endif; ?>
<?php if ($error): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert"><i class="fas fa-exclamation-triangle me-2"></i> <?= htmlspecialchars($error) ?><button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button></div>
<?php endif; ?>

<div class="row">
    <div class="col-lg-4 mb-4">
        <div class="card shadow">
            <div class="card-header py-3" style="background-color: #20c997; color: white;">
                <h6 class="m-0 font-weight-bold"><?= $category_to_edit ? 'Kateqoriyanı Dəyişdir' : 'Yeni Kateqoriya Əlavə Et' ?></h6>
            </div>
            <div class="card-body">
                <form action="categories.php" method="POST">
                    <input type="hidden" name="edit_id" value="<?= $category_to_edit ? $category_to_edit['id'] : '' ?>">
                    
                    <div class="mb-3">
                        <label for="name" class="form-label">Kateqoriya Adı</label>
                        <input type="text" class="form-control" id="name" name="name" 
                               value="<?= $category_to_edit ? htmlspecialchars($category_to_edit['name']) : '' ?>" required>
                    </div>

                    <div class="mb-3">
                        <label for="parent_id" class="form-label">Ana Kateqoriya (Əgər alt kateqoriyadırsa)</label>
                        <select class="form-select" id="parent_id" name="parent_id">
                            <option value="">-- Əsas Kateqoriya --</option>
                            <?php foreach ($parent_options as $parent): ?>
                                <option value="<?= $parent['id'] ?>" 
                                    <?= ($category_to_edit && $category_to_edit['parent_id'] == $parent['id']) ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($parent['name']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="mb-3">
                        <label for="sort_order" class="form-label">Sıralama Nömrəsi (1 = Ən öndə)</label>
                        <input type="number" min="1" class="form-control" id="sort_order" name="sort_order" 
                               value="<?= $category_to_edit ? htmlspecialchars($category_to_edit['sort_order']) : '99' ?>" required>
                    </div>
                    
                    <div class="mb-3 form-check">
                        <?php 
                            $is_active_checked = true; 
                            if ($category_to_edit) {
                                $is_active_checked = (bool)$category_to_edit['is_active'];
                            }
                        ?>
                        <input type="checkbox" class="form-check-input" id="is_active" name="is_active" 
                               value="1" <?= $is_active_checked ? 'checked' : '' ?>>
                        <label class="form-check-label" for="is_active">Kateqoriyanı Aktivləşdir</label>
                    </div>

                    <div class="d-grid">
                        <button type="submit" class="btn btn-dark">
                            <i class="fas <?= $category_to_edit ? 'fa-edit' : 'fa-plus-circle' ?> me-2"></i>
                            <?= $category_to_edit ? 'Yadda Saxla' : 'Kateqoriyanı Əlavə Et' ?>
                        </button>
                    </div>
                    <?php if ($category_to_edit): ?>
                        <div class="d-grid mt-2">
                            <a href="categories.php" class="btn btn-outline-secondary">Ləğv Et</a>
                        </div>
                    <?php endif; ?>
                </form>
            </div>
        </div>
    </div>

    <div class="col-lg-8">
        <div class="card shadow mb-4">
            <div class="card-header py-3">
                <h6 class="m-0 font-weight-bold text-dark">Bütün Kateqoriyalar</h6>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-hover" width="100%" cellspacing="0">
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Sıra</th> <th>Kateqoriya Adı</th>
                                <th>Ana Kateqoriya</th> <th>Status</th> <th>Yaradılma Tarixi</th> <th>Əməliyyatlar</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (!empty($categories)): ?>
                                <?php foreach ($categories as $category): ?>
                                    <tr>
                                        <td><?= $category['id'] ?></td>
                                        <td><span class="badge bg-primary"><?= $category['sort_order'] ?></span></td>
                                        <td><?= htmlspecialchars($category['name']) ?></td>
                                        <td>
                                            <?php if ($category['parent_id']): ?>
                                                <span class="badge bg-info text-dark"><?= htmlspecialchars($category['parent_name']) ?></span>
                                            <?php else: ?>
                                                <span class="badge bg-secondary">Əsas</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($category['is_active']): ?>
                                                <span class="badge bg-success"><i class="fas fa-check-circle"></i> Aktiv</span>
                                            <?php else: ?>
                                                <span class="badge bg-danger"><i class="fas fa-times-circle"></i> Deaktiv</span>
                                            <?php endif; ?>
                                        </td>
                                        <td><?= date('d.m.Y', strtotime($category['created_at'])) ?></td>
                                        <td>
                                            <a href="categories.php?action=edit&id=<?= $category['id'] ?>" class="btn btn-sm btn-info me-2" title="Dəyişdir">
                                                <i class="fas fa-edit"></i>
                                            </a>
                                            <a href="categories.php?action=delete&id=<?= $category['id'] ?>" 
                                               class="btn btn-sm btn-danger" 
                                               onclick="return confirm('Kateqoriya silinəcək. Əminsiniz?')"
                                               title="Sil">
                                                <i class="fas fa-trash-alt"></i>
                                            </a>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <tr>
                                    <td colspan="7" class="text-center text-muted">Hələ heç bir kateqoriya əlavə edilməyib.</td>
                                </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>

<?php
require_once '../includes/footer.php';
?>